
import requests
import xml.etree.ElementTree as ET
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QInputDialog
from qgis.core import QgsProject, QgsPointXY, QgsGeometry, QgsFeature, QgsVectorLayer, QgsField, QgsSymbol, QgsRendererCategory, QgsCategorizedSymbolRenderer, QgsCoordinateReferenceSystem, Qgis
from qgis.PyQt.QtCore import QVariant
from qgis.PyQt.QtGui import QColor

class AddressConverterPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        self.layer = None

    def initGui(self):
        self.action = QAction("住所から緯度経度変換", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Address Converter", self.action)

    def unload(self):
        self.iface.removeToolBarIcon(self.action)
        self.iface.removePluginMenu("&Address Converter", self.action)

    def run(self):
        # プロジェクトのCRSをEPSG:4326に設定
        crs = QgsCoordinateReferenceSystem("EPSG:4326")
        QgsProject.instance().setCrs(crs)
        
        # 現在のCRSを確認してログメッセージパネルに出力
        current_crs = QgsProject.instance().crs().authid()
        self.iface.messageBar().pushMessage("Debug", f"現在のプロジェクトのCRSは {current_crs} です", level=Qgis.Info)

        address, ok = QInputDialog.getText(self.iface.mainWindow(), "住所入力", "住所を入力してください:")
        if ok and address:
            result = self.convert_address_to_long_lat(address)
            if result != "0,0":
                lng, lat = map(float, result.split(","))
                self.add_point_to_map(lng, lat, address)
                QMessageBox.information(self.iface.mainWindow(), "経度と緯度", f"取得した経度と緯度: {result}")
                self.zoom_to_point(lng, lat)
            else:
                QMessageBox.warning(self.iface.mainWindow(), "エラー", "経度と緯度の取得に失敗しました。")

    def add_point_to_map(self, lng, lat, address):
        # レイヤーが存在しない場合、新しいレイヤーを作成
        if not self.layer:
            self.layer = QgsVectorLayer("Point?crs=EPSG:4326", "住所ポイント", "memory")
            provider = self.layer.dataProvider()
            provider.addAttributes([QgsField("name", QVariant.String)])
            self.layer.updateFields()
            QgsProject.instance().addMapLayer(self.layer)
            self.apply_style()

        # ポイントを追加
        point = QgsPointXY(lng, lat)
        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromPointXY(point))
        feature.setAttributes([address])  # フィールドに住所を設定
        self.layer.dataProvider().addFeatures([feature])
        self.layer.updateExtents()

    def apply_style(self):
        # ポイントのスタイルを赤色にし、サイズを小さくする
        symbol = QgsSymbol.defaultSymbol(self.layer.geometryType())
        symbol.setColor(QColor("red"))
        symbol.setSize(3)  # サイズを小さく設定

        renderer = QgsCategorizedSymbolRenderer("name", [QgsRendererCategory("", symbol, "住所ポイント")])
        self.layer.setRenderer(renderer)

    def zoom_to_point(self, lng, lat):
        # 指定された経度と緯度にズームする
        canvas = self.iface.mapCanvas()
        point = QgsPointXY(lng, lat)
        canvas.setCenter(point)
        canvas.zoomScale(1500)  # スケールを1500に設定してズーム
        canvas.refresh()

    @staticmethod
    def convert_address_to_long_lat(w_addr):
        url = f"http://www.geocoding.jp/api/?q={w_addr.strip()}"
        geo_code_result = "0,0"

        try:
            response = requests.get(url)
            if response.status_code == 200:
                xml_str = response.text
                xml_doc = ET.fromstring(xml_str)
                coordinate_element = xml_doc.find("coordinate")
                if coordinate_element is not None:
                    lat = coordinate_element.find("lat").text if coordinate_element.find("lat") is not None else None
                    lng = coordinate_element.find("lng").text if coordinate_element.find("lng") is not None else None
                    if lat and lng:
                        print(f"Debug: APIから取得した緯度: {lat}, 経度: {lng}")
                        geo_code_result = f"{lng},{lat}"
        except Exception as ex:
            print(f"An error occurred: {ex}")

        return geo_code_result
